% SOCIAL PLANNER VERSUS RAMSEY SETUP
% Options are:
%  1. SOCIAL_PLANNER or RAMSEY
%  2. NNCON (non-negativity constraint on investment)
%
% NOTE: 
% varrho*tau = MUX (optimal carbon tax: tau=MUX/varrho)
% MUX: shadow price of XX (in utility units)
% MUX_{t} = \beta (1 - \delta_X) MUX_{t+1}
%    + d'(X_t) ( \lambda_t [ dCES_dYb + dCES_dYg ]
%    - MUX_{t} \gamma_E \tilde{Y}_{b,t} ).

var
    C Y Yb Yg
    Kb Kg Ib Ig
    Lb Lg
    XX E dX
    Qb Qg
    Lambda varrho
    chi  
    // auxiliaries
    dCES_dYb dCES_dYg
    PhiYb PhiYg;

@#ifdef NNCON 
    var varepsilon;
@#endif

@#ifdef RAMSEY
    var tau;
@#endif

@#ifdef SOCIAL_PLANNER
    var dX_prime MUX;
@#endif

parameters
    betta eta
    A alfab alfag alfayg rhoy
    deltab deltag deltax
    gamaE
    dv_par dv_const clim_sen
    L_mom;

betta = 0.9975000000000001;
eta = 2;
alfab = 0.35;
alfag = 0.33;
rhoy = 2;
alfayg = 0.6301799628695098;
deltab = 0.025;
deltag = 0.025;
deltax = 0.9965000000000001;
gamaE = 3.872749316102583;
A = 0.8512300332534236;
L_mom = 1;
dv_par = 0.02;
clim_sen = 3;
dv_const = 0.00168;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% MODEL
model;
% Resource
Y = C + Ib + Ig;

% CES aggregator
Y = ( ((1-alfayg)^(1/rhoy)) * Yb^((rhoy-1)/rhoy) + (alfayg^(1/rhoy)) * Yg^((rhoy-1)/rhoy) )^(rhoy/(rhoy-1));

% Sectoral production with climate damage
Yb = (1 - dX) * A * (Kb(-1)^alfab) * (Lb^(1-alfab));
Yg = (1 - dX) * A * (Kg(-1)^alfag) * (Lg^(1-alfag));

% Capital accumulation
Kb = (1 - deltab)*Kb(-1) + Ib;
Kg = (1 - deltag)*Kg(-1) + Ig;

Lb + Lg = L_mom;

% Carbon stock and emissions
XX = deltax*XX(-1) + E;
E  = gamaE * Yb;

% Damage function (switches)
dX = 1 - exp( -dv_par/2 * (clim_sen*log(dv_const*XX)/log(2))^2 );

% Marg. ut. & SDF 
varrho = C^(-eta);
Lambda  = betta*varrho/varrho(-1);

% Marg. contribution Yi
dCES_dYb = (1-alfayg)^(1/rhoy) * (Y/Yb)^(1/rhoy);
dCES_dYg = (   alfayg)^(1/rhoy) * (Y/Yg)^(1/rhoy);

% Shadow values of sectoral outputs
PhiYg = varrho * dCES_dYg;
@#ifdef RAMSEY
    PhiYb = varrho * (dCES_dYb - tau*gamaE); 
@#endif 
@#ifdef SOCIAL_PLANNER
  	dX_prime = dv_par * (clim_sen*log(dv_const*XX)/log(2)) * (clim_sen/log(2)* (1/XX)) * exp( -dv_par/2 * (clim_sen*log(dv_const*XX)/log(2))^2 );
    MUX = betta*deltax*MUX(+1) + dX_prime * A * ( PhiYb * Kb(-1)^alfab * Lb^(1-alfab) + PhiYg * Kg(-1)^alfag * Lg^(1-alfag) );
    PhiYb = varrho * dCES_dYb - MUX*gamaE;
@#endif  

% Labor FOCs  one per sector 
chi = PhiYb * (1 - dX) * A * (1 - alfab) * Kb(-1)^alfab * Lb^(-alfab);
% equate labor demand for both sectors:
PhiYb * (1 - dX) * A * (1 - alfab) * Kb(-1)^alfab * Lb^(-alfab) = PhiYg * (1 - dX) * A * (1 - alfag) * Kg(-1)^alfag * Lg^(-alfag);

% Capital Tobin-Q recursions
Qb = betta*(1 - deltab)*Qb(+1) + betta*PhiYb(+1) * (1 - dX(+1)) * A(+1) * alfab * Kb^(alfab-1) * Lb(+1)^(1-alfab);
Qg = betta*(1 - deltag)*Qg(+1) + betta*PhiYg(+1) * (1 - dX(+1)) * A(+1) * alfag * Kg^(alfag-1) * Lg(+1)^(1-alfag);

% Non-negativity constraint?
@#ifdef NNCON           
    Qg = varrho; 
    Qb = varrho - varepsilon;
    [mcp = 'Ib > -1e-15']   
    varepsilon = 0;
@#else
   Qg = varrho; 
   Qb = varrho;

@#endif
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% INITIAL VALUES
initval;
  Y = 2.500000000000002;
  Yb = 1.007036521518217;
  Yg = 1.495759589187851;
  Kb = 12.27790432801828;
  Kg = 18.41685649202741;
  XX = 1138;
  E = 3.9;
  dX = 0.07565862258670031;
  C = 1.732630979498859;
  Ib = 0.306947608200457;
  Ig = 0.4604214123006853;
  Lb = 0.3788079470198687;
  Lg = 0.6211920529801314;
  Lambda = 0.9975000000000001;
  Qb = 0.3331101372030991;
  Qg = 0.3331101372030991;
  chi = 0.551531038570921;
  varrho = 0.3331101372030991;
@#ifdef RAMSEY
  tau = 0;
@#endif
@#ifdef SOCIAL_PLANNER
  MUX = 0;
@#endif
  
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% END VALUES
endval;
Kb = 8.508042539878646;
Kg = 20.5900426109516;
Lb = 0.274294578766714;
Lg = 0.725705421233286;
XX = 840.5977793802986;

@#ifdef RAMSEY
  tau = 0.06127812531577834;
@#endif
@#ifdef SOCIAL_PLANNER
  MUX = 0.01814491095642032;
@#endif

Ib = deltab*Kb;
Ig = deltag*Kg;

dX = 1 - exp( -dv_par/2 * (clim_sen*log(dv_const*XX)/log(2))^2 );

Yb = (1 - dX) * A * Kb^alfab * (Lb^(1-alfab));
Yg = (1 - dX) * A * Kg^alfag * (Lg^(1-alfag));
E  = gamaE * Yb;

Y = ( ((1-alfayg)^(1/rhoy)) * Yb^((rhoy-1)/rhoy) + (alfayg^(1/rhoy)) * Yg^((rhoy-1)/rhoy) )^(rhoy/(rhoy-1));
C = Y - Ib - Ig;

varrho = C^(-eta);
Lambda  = betta;

dCES_dYb = (1-alfayg)^(1/rhoy) * (Y/Yb)^(1/rhoy);
dCES_dYg = (   alfayg)^(1/rhoy) * (Y/Yg)^(1/rhoy);

PhiYg = varrho * dCES_dYg;

@#ifdef RAMSEY
    PhiYb = varrho * (dCES_dYb - tau*gamaE);  
@#endif
@#ifdef SOCIAL_PLANNER
    dX_prime = dv_par * (clim_sen*log(dv_const*XX)/log(2)) * (clim_sen/log(2)* (1/XX)) * exp( -dv_par/2 * (clim_sen*log(dv_const*XX)/log(2))^2 );
    PhiYb = varrho * dCES_dYb - MUX*gamaE;  
@#endif

chi = PhiYb * (1 - dX) * A * (1 - alfab) * Kb^alfab * Lb^(-alfab);

Qb = varrho;
Qg = varrho;

@#ifdef NNCON
     varepsilon = 0;
@#endif

end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SOLVE OPTIMUM under perfect foresight
@#ifdef RAMSEY
    planner_objective((C^(1-eta)-1)/(1-eta));
    ramsey_model(planner_discount=0.9975,instruments=(tau));
@#endif

steady (solve_algo=4, maxit=500);
resid;          
check;

perfect_foresight_setup(periods=2000);   
perfect_foresight_solver(lmmcp, maxit=200, stack_solve_algo=0, tolf=1e-8, tolx=1e-8);


